import tkinter as tk
from tkinter import ttk, messagebox, filedialog
import socket
import time
import threading
from datetime import datetime

# The user needs to make sure the IP Address of the PC and LDC50x on the same network.
# To change the LDC50x IP address, Subnet mask and Default Gateway, use command IPAD i, j, NMSK i, j and GWAY i,j through RS232 cable.

execution_error_code_mapping = {
    1: "Illegal value",
    2: "Wrong token",
    3: "Invalid bit",
    4: "Queue full",
    5: "Not compatible",
}

command_error_code_mapping = {
    1: "Illegal command",
    2: "Undefined command",
    3: "Illegal query",
    4: "Illegal set",
    5: "Missing parameter(s)",
    6: "Extra parameter(s)",
    7: "Null parameter(s)",
    8: "Parameter buffer overflow",
    9: "Bad floating-point",
    10: "Bad integer",
    11: "Bad integer token",
    12: "Bad token value",
    13: "Bad hex block",
    14: "Unknown token",
}


class LDC50X:
    """LDC50X Controller Class with Keep-Alive"""

    def __init__(self, ip_address, port=8888, timeout=5):
        self.ip_address = ip_address
        self.port = port
        self.timeout = timeout
        self.socket = None
        self.keepalive_thread = None
        self.running = False
        self.lock = threading.Lock()  # Thread safety for socket operations

    def connect(self):
        """Establish TCP connection to the LDC50X."""
        try:
            self.socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            self.socket.settimeout(self.timeout)

            # Enable TCP keep-alive
            self.socket.setsockopt(socket.SOL_SOCKET, socket.SO_KEEPALIVE, 1)

            # Windows-specific keep-alive settings (optional but recommended)
            try:
                # (enable, idle_time_ms, interval_ms)
                self.socket.ioctl(socket.SIO_KEEPALIVE_VALS, (1, 60000, 30000))
            except AttributeError:
                # Not on Windows, skip this
                pass

            self.socket.connect((self.ip_address, self.port))

            # Send blank line to initialize
            self.socket.sendall(b'\n')
            time.sleep(0.2)

            # Clear any initial response
            try:
                self.socket.recv(1024)
            except socket.timeout:
                pass

            # Unlock the ethernet command processor
            self.send_command("ULOC 1")
            time.sleep(0.1)

            # Start keep-alive thread
            self.running = True
            self.keepalive_thread = threading.Thread(target=self._keepalive_worker, daemon=True)
            self.keepalive_thread.start()

            return True
        except Exception as e:
            print(f"Connection failed: {e}")
            if self.socket:
                self.socket.close()
                self.socket = None
            return False

    def _keepalive_worker(self):
        """Background thread to send periodic queries to keep connection alive"""
        while self.running:
            time.sleep(120)  # Send keepalive every 2 minutes
            if not self.running:
                break
            try:
                with self.lock:
                    if self.socket:
                        # Send a simple query that doesn't affect device state
                        self.socket.sendall(b'*IDN?\n')
                        time.sleep(0.05)
                        try:
                            self.socket.recv(1024)  # Discard response
                        except socket.timeout:
                            pass
            except Exception as e:
                print(f"Keep-alive error: {e}")
                # Don't break - let the main thread handle reconnection

    def disconnect(self):
        """Close the TCP connection."""
        self.running = False
        if self.keepalive_thread:
            self.keepalive_thread.join(timeout=1.0)
        if self.socket:
            self.socket.close()
            self.socket = None

    def send_command(self, command):
        """Send a command to the LDC50X."""
        if not self.socket:
            raise Exception("Not connected to LDC50X")

        with self.lock:
            command = '\n' + command + '\n'  # add a Newline before and after every command to avoid LDC50x Ethernet parser error
            self.socket.sendall(command.encode('ascii'))
            time.sleep(0.05)
            # print({command})

    def query(self, command):
        """Send a query and receive response."""
        if not self.socket:
            raise Exception("Not connected to LDC50X")

        with self.lock:
            if not command.endswith('\n'):
                command += '\n'
            self.socket.sendall(command.encode('ascii'))
            time.sleep(0.05)
            response = self.socket.recv(1024).decode('ascii').strip()
            return response

    def send_custom_command(self, command):
        """Send a custom command and return response if it's a query"""
        if not self.socket:
            raise Exception("Not connected to LDC50X")

        # Check if it's a query command (contains '?')
        is_query = '?' in command

        if is_query:
            return self.query(command)
        else:
            self.send_command(command)
            return None

    # LDC50X Commands
    def laser_on(self):
        self.send_command("LDON 1")

    def laser_off(self):
        self.send_command("LDON 0")

    def get_laser_state(self):
        return int(self.query("LDON?"))

    def set_current(self, current_ma):
        self.send_command(f"SILD {current_ma}")

    def get_current(self):
        return float(self.query("SILD?"))

    def measure_voltage(self):
        return float(self.query("RVLD?"))

    def measure_photodiode(self):
        return float(self.query("RIPD?"))

    def measure_current(self):
        return float(self.query("RILD?"))

    def tec_on(self):
        self.send_command("TEON 1")

    def tec_off(self):
        self.send_command("TEON 0")

    def get_tec_state(self):
        return int(self.query("TEON?"))

    def set_temperature(self, temp_c):
        self.send_command(f"TEMP {temp_c}")

    def get_temperature(self):
        return float(self.query("TTRD?"))

    def measure_tec_voltage(self):
        return float(self.query("TVRD?"))

    def measure_tec_current(self):
        return float(self.query("TIRD?"))

    def set_current_range_high(self):
        self.send_command("RNGE 1")
        self.check_errors()

    def set_current_range_low(self):
        self.send_command("RNGE 0")
        self.check_errors()

    def get_current_range(self):
        return int(self.query("RNGE?"))

    def check_errors(self):
        execution_error_code = int(self.query("LEXE?"))
        command_error_code = int(self.query("LCME?"))
        error_queue = []
        if execution_error_code != 0:
            error_queue.append(f"Execution Error: {execution_error_code_mapping[execution_error_code]}")
        if command_error_code != 0:
            error_queue.append(f"Command Error: {command_error_code_mapping[command_error_code]}")
        if error_queue:
            raise RuntimeError(error_queue)


class LDC50X_GUI:
    def __init__(self, root):
        self.root = root
        self.root.title("LDC50x Laser Diode Controller Panel")
        self.root.geometry("500x900")

        self.ldc = None
        self.connected = False

        self.create_widgets()

        # Handle window close event
        self.root.protocol("WM_DELETE_WINDOW", self.on_closing)

    def create_widgets(self):
        # Connection Frame
        conn_frame = ttk.LabelFrame(self.root, text="Connection", padding=10)
        conn_frame.pack(fill="x", padx=10, pady=5)

        ttk.Label(conn_frame, text="IP Address:").grid(row=0, column=0, sticky="w", padx=5)
        self.ip_entry = ttk.Entry(conn_frame, width=20)
        self.ip_entry.insert(0, "172.25.96.149")
        self.ip_entry.grid(row=0, column=1, padx=5)

        self.connect_btn = ttk.Button(conn_frame, text="Connect", command=self.toggle_connection)
        self.connect_btn.grid(row=0, column=2, padx=5)

        self.status_label = ttk.Label(conn_frame, text="Status: Disconnected", foreground="red")
        self.status_label.grid(row=1, column=0, columnspan=3, pady=5)

        # Add informational text below status
        info_label = ttk.Label(conn_frame,
                               text="The LDC50x must be on the same network as your PC. Change its IP settings\n on front panel or via RS232 using commands: IPAD, NMSK, GWAY.",
                               foreground="blue",
                               font=("Arial", 8),
                               justify="center")
        info_label.grid(row=2, column=0, columnspan=3, pady=(0, 5))

        # Laser Diode Control Frame
        ld_frame = ttk.LabelFrame(self.root, text="Laser Diode Control", padding=10)
        ld_frame.pack(fill="x", padx=10, pady=5)

        # LD On/Off
        ld_btn_frame = ttk.Frame(ld_frame)
        ld_btn_frame.pack(fill="x", pady=5)

        self.ld_on_btn = ttk.Button(ld_btn_frame, text="LD ON", command=self.laser_on, state="disabled")
        self.ld_on_btn.pack(side="left", padx=5)

        self.ld_off_btn = ttk.Button(ld_btn_frame, text="LD OFF", command=self.laser_off, state="disabled")
        self.ld_off_btn.pack(side="left", padx=5)

        self.ld_state_label = ttk.Label(ld_btn_frame, text="State: Unknown", font=("Arial", 10, "bold"))
        self.ld_state_label.pack(side="left", padx=20)

        # Current Range Setting
        range_frame = ttk.Frame(ld_frame)
        range_frame.pack(fill="x", pady=5)

        ttk.Label(range_frame, text="Current Range:").pack(side="left", padx=5)

        self.range_high_btn = ttk.Button(range_frame, text="High Range", command=self.set_range_high, state="disabled")
        self.range_high_btn.pack(side="left", padx=5)

        self.range_low_btn = ttk.Button(range_frame, text="Low Range", command=self.set_range_low, state="disabled")
        self.range_low_btn.pack(side="left", padx=5)

        self.range_label = ttk.Label(range_frame, text="Range: Unknown", font=("Arial", 9))
        self.range_label.pack(side="left", padx=20)

        # LD Current Setting
        current_frame = ttk.Frame(ld_frame)
        current_frame.pack(fill="x", pady=5)

        ttk.Label(current_frame, text="Current (mA):").pack(side="left", padx=5)
        self.current_entry = ttk.Entry(current_frame, width=10)
        self.current_entry.insert(0, "---")
        self.current_entry.pack(side="left", padx=5)

        self.set_current_btn = ttk.Button(current_frame, text="Set Current", command=self.set_current, state="disabled")
        self.set_current_btn.pack(side="left", padx=5)

        # LD Readings
        reading_frame = ttk.Frame(ld_frame)
        reading_frame.pack(fill="x", pady=5)

        self.ld_current_label = ttk.Label(reading_frame, text="Current: -- mA")
        self.ld_current_label.pack(side="left", padx=5)

        self.ld_voltage_label = ttk.Label(reading_frame, text="Voltage: -- V")
        self.ld_voltage_label.pack(side="left", padx=5)

        self.ld_pd_label = ttk.Label(reading_frame, text="Photodiode: -- mA")
        self.ld_pd_label.pack(side="left", padx=5)

        # TEC Control Frame
        tec_frame = ttk.LabelFrame(self.root, text="TEC (Temperature) Control", padding=10)
        tec_frame.pack(fill="x", padx=10, pady=5)

        # TEC On/Off
        tec_btn_frame = ttk.Frame(tec_frame)
        tec_btn_frame.pack(fill="x", pady=5)

        self.tec_on_btn = ttk.Button(tec_btn_frame, text="TEC ON", command=self.tec_on, state="disabled")
        self.tec_on_btn.pack(side="left", padx=5)

        self.tec_off_btn = ttk.Button(tec_btn_frame, text="TEC OFF", command=self.tec_off, state="disabled")
        self.tec_off_btn.pack(side="left", padx=5)

        self.tec_state_label = ttk.Label(tec_btn_frame, text="State: Unknown", font=("Arial", 10, "bold"))
        self.tec_state_label.pack(side="left", padx=20)

        # Temperature Setting
        temp_frame = ttk.Frame(tec_frame)
        temp_frame.pack(fill="x", pady=5)

        ttk.Label(temp_frame, text="Temperature (°C):").pack(side="left", padx=5)
        self.temp_entry = ttk.Entry(temp_frame, width=10)
        self.temp_entry.insert(0, "---")
        self.temp_entry.pack(side="left", padx=5)

        self.set_temp_btn = ttk.Button(temp_frame, text="Set Temperature", command=self.set_temperature,
                                       state="disabled")
        self.set_temp_btn.pack(side="left", padx=5)

        # TEC Readings
        tec_reading_frame = ttk.Frame(tec_frame)
        tec_reading_frame.pack(fill="x", pady=5)

        self.tec_temp_label = ttk.Label(tec_reading_frame, text="Temperature: -- °C")
        self.tec_temp_label.pack(side="left", padx=5)

        self.tec_voltage_label = ttk.Label(tec_reading_frame, text="TEC Voltage: -- V")
        self.tec_voltage_label.pack(side="left", padx=5)

        self.tec_current_label = ttk.Label(tec_reading_frame, text="TEC Current: -- A")
        self.tec_current_label.pack(side="left", padx=5)

        # Send Any Command Frame (NEW SECTION)
        cmd_frame = ttk.LabelFrame(self.root, text="Send Any Command", padding=10)
        cmd_frame.pack(fill="x", padx=10, pady=5)

        # Command input frame
        cmd_input_frame = ttk.Frame(cmd_frame)
        cmd_input_frame.pack(fill="x", pady=5)

        ttk.Label(cmd_input_frame, text="Command:").pack(side="left", padx=5)
        self.custom_cmd_entry = ttk.Entry(cmd_input_frame, width=40)
        self.custom_cmd_entry.pack(side="left", padx=5)

        self.send_cmd_btn = ttk.Button(cmd_input_frame, text="Send Command", command=self.send_custom_command,
                                       state="disabled")
        self.send_cmd_btn.pack(side="left", padx=5)

        # Response display with text box
        response_label_frame = ttk.Frame(cmd_frame)
        response_label_frame.pack(fill="x", pady=(10, 2))
        ttk.Label(response_label_frame, text="Response:").pack(side="left", padx=5)

        self.cmd_response_text = tk.Text(cmd_frame, height=3, state="disabled", wrap="word")
        self.cmd_response_text.pack(fill="x", padx=5, pady=(0, 5))

        # Query Frame
        query_frame = ttk.LabelFrame(self.root, text="Query Measurements", padding=10)
        query_frame.pack(fill="x", padx=10, pady=5)

        self.query_btn = ttk.Button(query_frame, text="Refresh Readings", command=self.query_readings, state="disabled")
        self.query_btn.pack(pady=5)

        # Status/Log Frame
        log_frame = ttk.LabelFrame(self.root, text="Log", padding=10)
        log_frame.pack(fill="both", expand=True, padx=10, pady=5)

        # Log buttons frame
        log_btn_frame = ttk.Frame(log_frame)
        log_btn_frame.pack(fill="x", pady=(0, 5))

        self.save_log_btn = ttk.Button(log_btn_frame, text="Save Log", command=self.save_log)
        self.save_log_btn.pack(side="left", padx=5)

        self.clear_log_btn = ttk.Button(log_btn_frame, text="Clear Log", command=self.clear_log)
        self.clear_log_btn.pack(side="left", padx=5)

        self.log_text = tk.Text(log_frame, height=10, state="disabled")
        self.log_text.pack(fill="both", expand=True)

        scrollbar = ttk.Scrollbar(log_frame, command=self.log_text.yview)
        scrollbar.pack(side="right", fill="y")
        self.log_text.config(yscrollcommand=scrollbar.set)

    def log(self, message):
        """Add message to log with timestamp"""
        timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        log_entry = f"[{timestamp}] \n * {message}"
        self.log_text.config(state="normal")
        self.log_text.insert("end", f"{log_entry}\n")
        self.log_text.see("end")
        self.log_text.config(state="disabled")

    def save_log(self):
        """Save log content to a text file"""
        try:
            # Get current log content
            log_content = self.log_text.get("1.0", "end-1c")

            if not log_content.strip():
                messagebox.showinfo("Save Log", "Log is empty. Nothing to save.")
                return

            # Open file dialog
            default_filename = f"ldc50x_log_{datetime.now().strftime('%Y%m%d_%H%M%S')}.txt"
            filename = filedialog.asksaveasfilename(
                defaultextension=".txt",
                initialfile=default_filename,
                filetypes=[("Text files", "*.txt"), ("All files", "*.*")],
                title="Save Log File"
            )

            if filename:
                with open(filename, 'w') as f:
                    f.write(log_content)
                messagebox.showinfo("Save Log", f"Log saved successfully to:\n{filename}")
                self.log(f"Log saved to: {filename}")
        except Exception as e:
            messagebox.showerror("Save Log Error", f"Failed to save log:\n{str(e)}")

    def clear_log(self):
        """Clear the log content"""
        response = messagebox.askyesno("Clear Log", "Are you sure you want to clear the log?")
        if response:
            self.log_text.config(state="normal")
            self.log_text.delete("1.0", "end")
            self.log_text.config(state="disabled")
            self.log("Log cleared")

    def send_custom_command(self):
        """Send custom command entered by user"""
        try:
            command = self.custom_cmd_entry.get().strip()
            if not command:
                messagebox.showwarning("Warning", "Please enter a command")
                return

            self.log(f"Sending command: {command}")
            response = self.ldc.send_custom_command(command)

            # Update response text box
            self.cmd_response_text.config(state="normal")
            self.cmd_response_text.delete("1.0", "end")

            if response is not None:
                # It was a query command
                self.cmd_response_text.insert("1.0", response)
                self.log(f"Response: {response}")
            else:
                # It was a set command
                self.cmd_response_text.insert("1.0", "Command sent (no response)")
                self.log("Command sent successfully")

            self.cmd_response_text.config(state="disabled")

        except Exception as e:
            self.log(f"Error: {e}")
            self.cmd_response_text.config(state="normal")
            self.cmd_response_text.delete("1.0", "end")
            self.cmd_response_text.insert("1.0", f"Error: {str(e)}")
            self.cmd_response_text.config(state="disabled")
            messagebox.showerror("Error", str(e))

    def toggle_connection(self):
        if not self.connected:
            self.connect()
        else:
            self.disconnect()

    def connect(self):
        ip = self.ip_entry.get()
        self.log(f"Connecting to {ip}...")

        self.ldc = LDC50X(ip)
        if self.ldc.connect():
            self.connected = True
            self.status_label.config(text="Status: Connected", foreground="green")
            self.connect_btn.config(text="Disconnect")
            self.enable_controls()
            self.log("Connected successfully!")
            self.query_readings()
        else:
            self.log("Connection failed!")
            messagebox.showerror("Connection Error", "Failed to connect to LDC50X")

    def disconnect(self):
        if self.ldc:
            self.ldc.disconnect()
            self.connected = False
            self.status_label.config(text="Status: Disconnected", foreground="red")
            self.connect_btn.config(text="Connect")
            self.disable_controls()
            self.log("Disconnected")

    def enable_controls(self):
        self.ld_on_btn.config(state="normal")
        self.ld_off_btn.config(state="normal")
        self.set_current_btn.config(state="normal")
        self.range_high_btn.config(state="normal")
        self.range_low_btn.config(state="normal")
        self.tec_on_btn.config(state="normal")
        self.tec_off_btn.config(state="normal")
        self.set_temp_btn.config(state="normal")
        self.query_btn.config(state="normal")
        self.send_cmd_btn.config(state="normal")

    def disable_controls(self):
        self.ld_on_btn.config(state="disabled")
        self.ld_off_btn.config(state="disabled")
        self.set_current_btn.config(state="disabled")
        self.range_high_btn.config(state="disabled")
        self.range_low_btn.config(state="disabled")
        self.tec_on_btn.config(state="disabled")
        self.tec_off_btn.config(state="disabled")
        self.set_temp_btn.config(state="disabled")
        self.query_btn.config(state="disabled")
        self.send_cmd_btn.config(state="disabled")

    def laser_on(self):
        try:
            self.ldc.laser_on()
            self.log("Laser turned ON")
            self.update_laser_state()
        except Exception as e:
            self.log(f"Error: {e}")
            messagebox.showerror("Error", str(e))

    def laser_off(self):
        try:
            self.ldc.laser_off()
            self.log("Laser turned OFF")
            self.update_laser_state()
        except Exception as e:
            self.log(f"Error: {e}")
            messagebox.showerror("Error", str(e))

    def set_current(self):
        try:
            current = float(self.current_entry.get())
            self.ldc.set_current(current)
            self.log(f"Current set to {current} mA")
            time.sleep(0.1)
            actual = self.ldc.get_current()
            self.log(f"Current confirmed: {actual} mA")
        except ValueError:
            messagebox.showerror("Error", "Invalid current value")
        except Exception as e:
            self.log(f"Error: {e}")
            messagebox.showerror("Error", str(e))

    def tec_on(self):
        try:
            self.ldc.tec_on()
            self.log("TEC turned ON")
            self.update_tec_state()
        except Exception as e:
            self.log(f"Error: {e}")
            messagebox.showerror("Error", str(e))

    def tec_off(self):
        try:
            self.ldc.tec_off()
            self.log("TEC turned OFF")
            self.update_tec_state()
        except Exception as e:
            self.log(f"Error: {e}")
            messagebox.showerror("Error", str(e))

    def set_temperature(self):
        try:
            temp = float(self.temp_entry.get())
            self.ldc.set_temperature(temp)
            self.log(f"Temperature set to {temp} °C")
        except ValueError:
            messagebox.showerror("Error", "Invalid temperature value")
        except Exception as e:
            self.log(f"Error: {e}")
            messagebox.showerror("Error", str(e))

    def set_range_high(self):
        try:
            self.ldc.set_current_range_high()
            self.log("Current range set to HIGH")
            self.update_current_range()
        except Exception as e:
            self.log(f"Error: {e}")
            messagebox.showerror("Error", str(e))

    def set_range_low(self):
        try:
            self.ldc.set_current_range_low()
            self.log("Current range set to LOW")
            self.update_current_range()
        except Exception as e:
            self.log(f"Error: {e}")
            messagebox.showerror("Error", str(e))

    def update_current_range(self):
        try:
            range_val = self.ldc.get_current_range()
            if range_val:
                self.range_label.config(text="Range: HIGH", foreground="blue")
            else:
                self.range_label.config(text="Range: LOW", foreground="orange")
        except Exception as e:
            self.log(f"Error reading current range: {e}")

    def update_laser_state(self):
        try:
            state = self.ldc.get_laser_state()
            if state:
                self.ld_state_label.config(text="State: ON", foreground="green")
            else:
                self.ld_state_label.config(text="State: OFF", foreground="red")
        except Exception as e:
            self.log(f"Error reading laser state: {e}")

    def update_tec_state(self):
        try:
            state = self.ldc.get_tec_state()
            if state:
                self.tec_state_label.config(text="State: ON", foreground="green")
            else:
                self.tec_state_label.config(text="State: OFF", foreground="red")
        except Exception as e:
            self.log(f"Error reading TEC state: {e}")

    def query_readings(self):
        try:
            # Update states
            self.update_laser_state()
            self.update_tec_state()
            self.update_current_range()

            # Query LD readings
            current_set = self.ldc.get_current()
            current_actual = self.ldc.measure_current()
            voltage = self.ldc.measure_voltage()
            pd = self.ldc.measure_photodiode()

            self.ld_current_label.config(text=f"Current: {current_actual:.3f} mA")
            self.ld_voltage_label.config(text=f"Voltage: {voltage:.3f} V")
            self.ld_pd_label.config(text=f"Photodiode: {pd:.3f} mA")

            # Query TEC readings
            rtemp = self.ldc.get_temperature()
            tec_voltage = self.ldc.measure_tec_voltage()
            tec_current = self.ldc.measure_tec_current()
            self.tec_temp_label.config(text=f"Temperature: {rtemp:.2f} °C")
            self.tec_voltage_label.config(text=f"TEC Voltage: {tec_voltage:.3f} V")
            self.tec_current_label.config(text=f"TEC Current: {tec_current:.3f} A")

            self.log(
                f"Readings: "
                f"\n   LD: ILD={current_actual:.3f}mA, VLD={voltage:.3f}V, IPD={pd:.3f}mA "
                f"\n   TE: Tte={rtemp:.2f}°C, Ite={tec_current:.3f}A, Vte={tec_voltage:.3f}V")

        except Exception as e:
            self.log(f"Error querying readings: {e}")
            messagebox.showerror("Error", str(e))

    def on_closing(self):
        """Handle window close event"""
        if self.connected:
            self.disconnect()
        self.root.destroy()


if __name__ == "__main__":
    root = tk.Tk()
    app = LDC50X_GUI(root)
    root.mainloop()